/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.CascadingValueProvider;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldValueProvider;
import com.inet.report.Fields;
import com.inet.report.PromptField;
import com.inet.report.SortField;

/**
 * This sample shows you how to use cascading prompt fields with rdc. Cascading Prompts are prompts whose default values
 * are generated dynamically at run-time and whose default values depend on the value of a parent prompt, rather than
 * set to a number of static values which never change. To create a cascading prompt, you must simply call
 * {@link PromptField#setDefaultValueProvider(com.inet.report.DynamicValueProvider)} and set it to a
 * {@link CascadingValueProvider}.
 */
public class CascadingPromptsSample extends RDCSample {

    // Change this before running the sample if the location of the Report
    // "mainrep.rpt" which comes with the rdc-samples has changed
    private static final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Creates a new engine, set the report file and add parameter fields and record selection formula using it to the
     * report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            // create a new Engine
            Engine eng = new Engine( exportFmt );
            // set the name of the report file that contains the report
            URL reportURL = getClass().getResource( '/' + MAINREPORT );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( MAINREPORT );
            }
            Fields fields = eng.getFields();

            // add a prompt field to fields set
            PromptField countryPrompt = fields.addPromptField( "country", "Customers from which country?", Field.STRING );
            // add another prompt field. This will be our cascading prompt field, since our choice
            // of city should depend on which country the user chooses.
            PromptField cityPrompt = fields.addPromptField( "city", "Customers from which city?", Field.STRING );

            // First we create a simple dynamic value provider which sets the default values for
            // the country prompt at run-time by fetching the country values of the records in
            // the "Country" column.
            FieldValueProvider provider = new FieldValueProvider( fields.getDatabaseField( "Customers.Country" ), null, SortField.DESCENDING_ORDER );
            countryPrompt.setDefaultValueProvider( provider );

            // Now we create our cascading value provider, linking it to its parent prompt.
            CascadingValueProvider cascadingProvider =
                new CascadingValueProvider( countryPrompt, provider, fields.getDatabaseField( "Customers.City" ), null, SortField.DESCENDING_ORDER );
            cityPrompt.setDefaultValueProvider( cascadingProvider );

            eng.setSF( "{Customers.Country} = {?country} and ({Customers.City} = {?city} or {?city} = '')" );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new CascadingPromptsSample().initUI();
    }
}
